<?php
/**
 * Database handler for Image Compression Plugin
 */

if (!defined('ABSPATH')) {
    exit;
}

class ICP_Database {
    
    public static function create_tables() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'icp_compression_stats';
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE $table_name (
            id int(11) NOT NULL AUTO_INCREMENT,
            attachment_id int(11) NOT NULL,
            original_size int(11) NOT NULL,
            compressed_size int(11) NOT NULL,
            savings_percent float NOT NULL,
            compression_date datetime DEFAULT CURRENT_TIMESTAMP,
            status varchar(20) DEFAULT 'success',
            error_message text DEFAULT NULL,
            original_path varchar(255) DEFAULT NULL,
            backup_path varchar(255) DEFAULT NULL,
            is_restored tinyint(1) DEFAULT 0,
            restore_date datetime DEFAULT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY attachment_id (attachment_id)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    public static function save_compression_data($data) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'icp_compression_stats';
        
        return $wpdb->replace(
            $table_name,
            array(
                'attachment_id' => $data['attachment_id'],
                'original_size' => $data['original_size'],
                'compressed_size' => $data['compressed_size'],
                'savings_percent' => $data['savings_percent'],
                'status' => $data['status'],
                'error_message' => isset($data['error_message']) ? $data['error_message'] : null,
                'original_path' => isset($data['original_path']) ? $data['original_path'] : null,
                'backup_path' => isset($data['backup_path']) ? $data['backup_path'] : null,
                'is_restored' => isset($data['is_restored']) ? $data['is_restored'] : 0,
                'restore_date' => isset($data['restore_date']) ? $data['restore_date'] : null
            ),
            array('%d', '%d', '%d', '%f', '%s', '%s', '%s', '%s', '%d', '%s')
        );
    }
    
    public static function get_compression_stats($attachment_id = null) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'icp_compression_stats';
        
        if ($attachment_id) {
            return $wpdb->get_row(
                $wpdb->prepare("SELECT * FROM $table_name WHERE attachment_id = %d", $attachment_id)
            );
        } else {
            return $wpdb->get_results("SELECT * FROM $table_name ORDER BY compression_date DESC");
        }
    }
    
    public static function get_compression_stats_paginated($args = array()) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'icp_compression_stats';
        
        // Default arguments
        $defaults = array(
            'per_page' => 10,
            'page' => 1,
            'search' => '',
            'status' => '',
            'date_from' => '',
            'date_to' => '',
            'orderby' => 'compression_date',
            'order' => 'DESC'
        );
        
        $args = wp_parse_args($args, $defaults);
        
        // Build WHERE clause
        $where_conditions = array();
        $prepare_values = array();
        
        // Search functionality
        if (!empty($args['search'])) {
            // Join with posts table to search in image names
            $search_term = '%' . $wpdb->esc_like($args['search']) . '%';
            $where_conditions[] = "(p.post_title LIKE %s OR s.attachment_id LIKE %s)";
            $prepare_values[] = $search_term;
            $prepare_values[] = $search_term;
        }
        
        // Status filter
        if (!empty($args['status'])) {
            $where_conditions[] = "s.status = %s";
            $prepare_values[] = $args['status'];
        }
        
        // Date range filters
        if (!empty($args['date_from'])) {
            $where_conditions[] = "s.compression_date >= %s";
            $prepare_values[] = $args['date_from'] . ' 00:00:00';
        }
        
        if (!empty($args['date_to'])) {
            $where_conditions[] = "s.compression_date <= %s";
            $prepare_values[] = $args['date_to'] . ' 23:59:59';
        }
        
        $where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';
        
        // Validate orderby
        $allowed_orderby = array('compression_date', 'original_size', 'compressed_size', 'savings_percent', 'attachment_id');
        if (!in_array($args['orderby'], $allowed_orderby)) {
            $args['orderby'] = 'compression_date';
        }
        
        // Validate order
        $args['order'] = strtoupper($args['order']) === 'ASC' ? 'ASC' : 'DESC';
        
        // Calculate offset
        $offset = ($args['page'] - 1) * $args['per_page'];
        
        // Build the query
        $sql = "
            SELECT s.*, p.post_title as image_name
            FROM $table_name s
            INNER JOIN {$wpdb->posts} p ON s.attachment_id = p.ID AND p.post_type = 'attachment'
            $where_clause
            ORDER BY s.{$args['orderby']} {$args['order']}
            LIMIT %d OFFSET %d
        ";
        
        $prepare_values[] = $args['per_page'];
        $prepare_values[] = $offset;
        
        if (!empty($prepare_values)) {
            $results = $wpdb->get_results($wpdb->prepare($sql, $prepare_values));
        } else {
            $results = $wpdb->get_results($sql);
        }
        
        // Get total count for pagination
        $count_sql = "
            SELECT COUNT(*)
            FROM $table_name s
            INNER JOIN {$wpdb->posts} p ON s.attachment_id = p.ID AND p.post_type = 'attachment'
            $where_clause
        ";
        
        if (!empty($where_conditions)) {
            $count_prepare_values = array_slice($prepare_values, 0, -2); // Remove LIMIT and OFFSET values
            $total_items = $wpdb->get_var($wpdb->prepare($count_sql, $count_prepare_values));
        } else {
            $total_items = $wpdb->get_var($count_sql);
        }
        
        return array(
            'items' => $results,
            'total_items' => (int) $total_items,
            'total_pages' => ceil($total_items / $args['per_page']),
            'current_page' => $args['page'],
            'per_page' => $args['per_page']
        );
    }
    
    public static function get_available_statuses() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'icp_compression_stats';
        
        return $wpdb->get_col("SELECT DISTINCT status FROM $table_name ORDER BY status");
    }
    
    public static function get_total_savings() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'icp_compression_stats';

        $result = $wpdb->get_row("
            SELECT
                COUNT(*) as total_images,
                SUM(original_size) as total_original_size,
                SUM(compressed_size) as total_compressed_size,
                SUM(original_size - compressed_size) as total_savings
            FROM $table_name
            WHERE status = 'success' AND (is_restored IS NULL OR is_restored = 0)
        ");

        return $result;
    }
    
    public static function get_uncompressed_images($excluded_folders = array()) {
        global $wpdb;

        $stats_table = $wpdb->prefix . 'icp_compression_stats';
        
        // Base query to get uncompressed images (including restored images)
        $sql = "
            SELECT p.ID, p.post_title, pm.meta_value as file_path, s.status as compression_status
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_wp_attached_file'
            LEFT JOIN $stats_table s ON p.ID = s.attachment_id
            WHERE p.post_type = 'attachment'
            AND p.post_mime_type IN ('image/jpeg', 'image/png', 'image/gif', 'image/webp')
            AND (s.attachment_id IS NULL OR s.status = 'failed' OR s.is_restored = 1)
        ";
        
        // If we have excluded folders and FileBird is active, modify the query
        if (!empty($excluded_folders) && self::is_filebird_active()) {
            // Get attachment IDs that are in excluded folders
            $excluded_attachment_ids = self::get_attachments_in_folders($excluded_folders);
            
            if (!empty($excluded_attachment_ids)) {
                $placeholders = implode(',', array_fill(0, count($excluded_attachment_ids), '%d'));
                $sql .= " AND p.ID NOT IN ($placeholders)";
            }
        }
        
        $sql .= " ORDER BY p.post_date DESC";
        
        // Execute query with excluded attachment IDs if needed
        if (!empty($excluded_folders) && self::is_filebird_active() && !empty($excluded_attachment_ids)) {
            return $wpdb->get_results($wpdb->prepare($sql, $excluded_attachment_ids));
        } else {
            return $wpdb->get_results($sql);
        }
    }
    
    public static function get_failed_compression_images($excluded_folders = array()) {
        global $wpdb;

        $stats_table = $wpdb->prefix . 'icp_compression_stats';

        $sql = "
            SELECT p.ID, p.post_title, pm.meta_value as file_path, s.error_message, s.compression_date
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_wp_attached_file'
            INNER JOIN $stats_table s ON p.ID = s.attachment_id
            WHERE p.post_type = 'attachment'
            AND p.post_mime_type IN ('image/jpeg', 'image/png', 'image/gif', 'image/webp')
            AND s.status = 'failed'
        ";

        // If we have excluded folders and FileBird is active, modify the query
        if (!empty($excluded_folders) && self::is_filebird_active()) {
            // Get attachment IDs that are in excluded folders
            $excluded_attachment_ids = self::get_attachments_in_folders($excluded_folders);

            if (!empty($excluded_attachment_ids)) {
                $placeholders = implode(',', array_fill(0, count($excluded_attachment_ids), '%d'));
                $sql .= " AND p.ID NOT IN ($placeholders)";
            }
        }

        $sql .= " ORDER BY s.compression_date DESC";

        // Execute query with excluded attachment IDs if needed
        if (!empty($excluded_folders) && self::is_filebird_active() && !empty($excluded_attachment_ids)) {
            return $wpdb->get_results($wpdb->prepare($sql, $excluded_attachment_ids));
        } else {
            return $wpdb->get_results($sql);
        }
    }
    
    public static function get_total_compression_errors() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'icp_compression_stats';
        
        $count = $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE status = 'failed'");
        
        return intval($count);
    }
    
    public static function delete_compression_data($attachment_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'icp_compression_stats';
        
        return $wpdb->delete(
            $table_name,
            array('attachment_id' => $attachment_id),
            array('%d')
        );
    }
    
    public static function reset_stats() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'icp_compression_stats';
        
        return $wpdb->query("TRUNCATE TABLE $table_name");
    }
    
    public static function reset_failed_compressions() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'icp_compression_stats';
        
        return $wpdb->delete(
            $table_name,
            array('status' => 'failed'),
            array('%s')
        );
    }
    
    /**
     * Clean up orphaned compression records for deleted attachments
     */
    public static function cleanup_orphaned_records() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'icp_compression_stats';
        
        // Delete compression records where the attachment no longer exists
        $deleted_count = $wpdb->query("
            DELETE s FROM $table_name s
            LEFT JOIN {$wpdb->posts} p ON s.attachment_id = p.ID
            WHERE p.ID IS NULL OR p.post_type != 'attachment'
        ");
        
        return $deleted_count;
    }
    
    /**
     * Check if FileBird plugin is active and its tables exist
     */
    public static function is_filebird_active() {
        global $wpdb;
        
        // Ensure plugin functions are available
        if (!function_exists('is_plugin_active')) {
            require_once(ABSPATH . 'wp-admin/includes/plugin.php');
        }
        
        // Check if FileBird plugin is active
        if (!is_plugin_active('filebird/filebird.php') && !is_plugin_active('filebird-pro/filebird-pro.php')) {
            return false;
        }
        
        // Check if FileBird tables exist
        $folder_table = $wpdb->prefix . 'fbv';
        $relation_table = $wpdb->prefix . 'fbv_attachment_folder';
        
        $folder_table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $folder_table)) === $folder_table;
        $relation_table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $relation_table)) === $relation_table;
        
        return $folder_table_exists && $relation_table_exists;
    }
    
    /**
     * Get all FileBird folders
     */
    public static function get_filebird_folders() {
        global $wpdb;
        
        if (!self::is_filebird_active()) {
            return array();
        }
        
        $folder_table = $wpdb->prefix . 'fbv';
        
        $folders = $wpdb->get_results("
            SELECT id, name, parent 
            FROM $folder_table 
            ORDER BY parent ASC, name ASC
        ");
        
        return $folders;
    }
    
    /**
     * Get attachment IDs that belong to specific folders (including subfolders)
     */
    public static function get_attachments_in_folders($folder_ids) {
        global $wpdb;
        
        if (!self::is_filebird_active() || empty($folder_ids)) {
            return array();
        }
        
        // Get all folder IDs including subfolders
        $all_folder_ids = self::get_all_subfolder_ids($folder_ids);
        
        if (empty($all_folder_ids)) {
            return array();
        }
        
        $relation_table = $wpdb->prefix . 'fbv_attachment_folder';
        $placeholders = implode(',', array_fill(0, count($all_folder_ids), '%d'));
        
        $attachment_ids = $wpdb->get_col($wpdb->prepare("
            SELECT DISTINCT attachment_id 
            FROM $relation_table 
            WHERE folder_id IN ($placeholders)
        ", $all_folder_ids));
        
        return array_map('intval', $attachment_ids);
    }
    
    /**
     * Get all subfolder IDs recursively for given folder IDs
     */
    public static function get_all_subfolder_ids($folder_ids) {
        global $wpdb;
        
        if (!self::is_filebird_active() || empty($folder_ids)) {
            return array();
        }
        
        $folder_table = $wpdb->prefix . 'fbv';
        $all_ids = array_map('intval', $folder_ids);
        $processed_ids = array();
        
        // Keep looking for subfolders until no new ones are found
        while (!empty($all_ids)) {
            $current_batch = array_diff($all_ids, $processed_ids);
            if (empty($current_batch)) {
                break;
            }
            
            $processed_ids = array_merge($processed_ids, $current_batch);
            $placeholders = implode(',', array_fill(0, count($current_batch), '%d'));
            
            $subfolder_ids = $wpdb->get_col($wpdb->prepare("
                SELECT id 
                FROM $folder_table 
                WHERE parent IN ($placeholders)
            ", $current_batch));
            
            $all_ids = array_merge($all_ids, array_map('intval', $subfolder_ids));
            $all_ids = array_unique($all_ids);
        }
        
        return $processed_ids;
    }
    
    /**
     * Get folder hierarchy as nested array for display
     */
    public static function get_filebird_folder_hierarchy() {
        $folders = self::get_filebird_folders();
        
        if (empty($folders)) {
            return array();
        }
        
        // Convert to associative array for easier processing
        $folder_map = array();
        foreach ($folders as $folder) {
            $folder_map[$folder->id] = array(
                'id' => $folder->id,
                'name' => $folder->name,
                'parent' => $folder->parent,
                'children' => array()
            );
        }
        
        // Build hierarchy
        $hierarchy = array();
        foreach ($folder_map as $folder) {
            if ($folder['parent'] == 0) {
                $hierarchy[$folder['id']] = $folder;
            } else {
                if (isset($folder_map[$folder['parent']])) {
                    $folder_map[$folder['parent']]['children'][$folder['id']] = $folder;
                }
            }
        }
        
        return $hierarchy;
    }

    /**
     * Update restore status for an attachment
     */
    public static function update_restore_status($attachment_id, $is_restored = true) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'icp_compression_stats';

        return $wpdb->update(
            $table_name,
            array(
                'is_restored' => $is_restored ? 1 : 0,
                'restore_date' => $is_restored ? current_time('mysql') : null
            ),
            array('attachment_id' => $attachment_id),
            array('%d', '%s'),
            array('%d')
        );
    }

    /**
     * Get images that can be restored (have backup files)
     */
    public static function get_restorable_images($limit = null, $offset = 0) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'icp_compression_stats';

        $sql = "
            SELECT s.*, p.post_title, pm.meta_value as file_path
            FROM $table_name s
            LEFT JOIN {$wpdb->posts} p ON s.attachment_id = p.ID
            LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_wp_attached_file'
            WHERE s.backup_path IS NOT NULL
            AND s.backup_path != ''
            AND s.status = 'success'
            AND p.post_type = 'attachment'
            ORDER BY s.compression_date DESC
        ";

        if ($limit) {
            $sql .= $wpdb->prepare(" LIMIT %d OFFSET %d", $limit, $offset);
        }

        return $wpdb->get_results($sql);
    }

    /**
     * Get count of restorable images
     */
    public static function get_restorable_images_count() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'icp_compression_stats';

        return $wpdb->get_var("
            SELECT COUNT(*)
            FROM $table_name s
            LEFT JOIN {$wpdb->posts} p ON s.attachment_id = p.ID
            WHERE s.backup_path IS NOT NULL
            AND s.backup_path != ''
            AND s.status = 'success'
            AND p.post_type = 'attachment'
        ");
    }
}